/*****************************************************************************
 * nav.c: wrappers to libdvdread read functions.
 *****************************************************************************
 * Copyright (C) 2002 VideoLAN
 * $Id: nav.c,v 1.10 2003/01/30 09:58:05 sam Exp $
 *
 * Authors: Stphane Borel <stef@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/

#include "config.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#   include <unistd.h>
#endif

#include "common.h"

#include <dvdread/ifo_types.h>
#include <dvdread/nav_read.h>

#include "dvdplay/dvdplay.h"
#include "dvdplay/nav.h"

#include "tools.h"
#include "command.h"
#include "vmg.h"
#include "msg.h"

/*
 * Local prototypes
 */
static int ProcessPCI    ( dvdplay_ptr dvdplay );
static int MouseButton   ( pci_t * p_pci, dvdplay_ctrl_mouse_t * p_mouse );
static int NextDataPacket( const dvdplay_ptr dvdplay, byte_t** pp_buffer );
static int ReadNav       ( dvdplay_ptr dvdplay, byte_t* p_buffer );

#define CELL \
    dvdplay->state.p_pgc->cell_playback[dvdplay->state.i_cellN - 1 ]

/*****************************************************************************
 * dvdplay_nav: read navigation block to update state.
 *****************************************************************************
 * The state has to be ready to read a navigation block (no seek is done)
 *****************************************************************************/
int dvdplay_nav( dvdplay_ptr dvdplay )
{
    byte_t      p_buffer[DVD_VIDEO_LB_LEN];
    int         i_block;

    _dvdplay_dbg( dvdplay, "retrieving navigation packet" );

    i_block = dvdplay->state.p_pgc->cell_playback[
                dvdplay->state.i_cellN - 1  ].first_sector
            + dvdplay->state.i_blockN;

    if( DVDReadBlocks( dvdplay->p_file, i_block, 1, p_buffer ) != 1 )
    {
        _dvdplay_err( dvdplay, "cannot get navigation packet in block %d",
                               i_block );
    }

    return ReadNav( dvdplay, p_buffer );
}

/*****************************************************************************
 * dvdplay_read: read maximum i_count bytes and update state/registers.
 *****************************************************************************
 * Most of the navigation work is done from here: the jumps between cells,
 * still time... Therefore the callback is used much to inform the client
 * appplication.
 *****************************************************************************/
int dvdplay_read( dvdplay_ptr dvdplay,
                         byte_t* p_buffer, int i_count )
{
    int                 i_block;
    int                 i_last;
    int                 i_try;
    int                 i_read = 0;

    /* A jump has occurred from outside (button ...) */
    if( dvdplay->b_jump )
    {
        _dvdplay_dbg( dvdplay, "jumping..." );

        dvdplay->pf_callback( dvdplay->p_args, JUMP );
        dvdplay_nav( dvdplay );
        dvdplay->b_jump = 0;
    }

    /* Pending block */
    i_block = CELL.first_sector + dvdplay->state.i_blockN;

    /* Last block in vobu */
    i_last = dvdplay->dsi.dsi_gi.nv_pck_lbn + dvdplay->dsi.dsi_gi.vobu_ea;

    i_try = i_last - i_block + 1;

    if( i_try <= 0 )
    {
        if( i_try )
        {
            _dvdplay_warn( dvdplay, "current block is not the last one in vobu"
                                    " or cell %d", i_try );
        }

        i_block = dvdplay->dsi.dsi_gi.nv_pck_lbn +
                 ( dvdplay->dsi.vobu_sri.next_vobu & 0x3fffffff );

        if( i_block > CELL.last_vobu_start_sector )
        {
            /* New cell */
            if( dvdplay_next_cell( dvdplay ) < 0)
            {
                _dvdplay_err( dvdplay, "read for new cell failed in block %d",
                                       i_block );
                return -1;
            }
            i_block = CELL.first_sector + dvdplay->state.i_blockN;

            dvdplay->b_jump = 0;
        }

        if( DVDReadBlocks( dvdplay->p_file, i_block, 1, p_buffer ) != 1 )
        {
            _dvdplay_err( dvdplay, "read for new vobu failed in block %d",
                                   i_block );
            return -1;
        }

        ReadNav( dvdplay, p_buffer );
        p_buffer += DVD_VIDEO_LB_LEN;
        i_read++;
        i_block++;
        i_count--;

        i_last = dvdplay->dsi.dsi_gi.nv_pck_lbn + dvdplay->dsi.dsi_gi.vobu_ea;
        i_try = i_last - i_block + 1;

        if( (dvdplay->dsi.vobu_sri.next_vobu & 0x80000000) == 0
                && dvdplay->dsi.dsi_gi.vobu_1stref_ea != 0 )
        {
            _dvdplay_dbg( dvdplay, "complete video in vobu" );

            dvdplay->pf_callback( dvdplay->p_args, COMPLETE_VIDEO );
        }
    }

    if( i_try > i_count )
    {
        i_try = i_count;
    }

    if( DVDReadBlocks( dvdplay->p_file, i_block, i_try, p_buffer ) != i_try )
    {
        _dvdplay_err( dvdplay, "read for %d failed in block %d",
                               i_try, i_block );
        return -1;
    }

    i_read += i_try;
    i_block += i_try;

    dvdplay->state.i_blockN = i_block - CELL.first_sector;

    if( ( i_last - i_block + 1 ) <= 0 )
    {
        //_dvdplay_dbg( dvdplay, "end of cell %d", dvdplay->state.i_cellN );

        dvdplay->pf_callback( dvdplay->p_args, END_OF_CELL );

        if( CELL.still_time )
        {
            _dvdplay_dbg( dvdplay, "still time %d", CELL.still_time );
            dvdplay->pf_callback( dvdplay->p_args, STILL_TIME );
        }
    }

    //_dvdplay_dbg( dvdplay, "read %d blocks", i_read );

    return i_read;
}

/*****************************************************************************
 * dvdplay_seek: seek to i_off byte from title start
 *****************************************************************************
 * The function also update state to allow further reading,
 * and sets the new chapter
 *****************************************************************************/
int dvdplay_seek( dvdplay_ptr dvdplay, int i_off )
{
    pgc_t*  p_pgc;
    int     i_pgc_block;
    int     i_block;
    int     i_cell;
    int     i_vobu;
    int     i_admap_entries;

    p_pgc = dvdplay->state.p_pgc;

    i_pgc_block = dvdplay_title_first( dvdplay );
    i_block = i_pgc_block + i_off;

    /* find cell playback number */
    i_cell = 1;
    while( i_cell <= p_pgc->nr_of_cells &&
           p_pgc->cell_playback[i_cell-1].last_sector < i_block )
    {
        ++i_cell;
    }

    if( i_cell > p_pgc->nr_of_cells )
    {
        _dvdplay_err( dvdplay, "seeking to block %d failed (nonexistent block)",
                               i_block );
        return -1;
    }

    /* find vobu containing our block */
    i_vobu = 1;
    i_admap_entries = ( dvdplay->p_vtsi->vts_vobu_admap->last_byte + 1
                        - VOBU_ADMAP_SIZE ) / 4;
    while( i_vobu < i_admap_entries &&
           dvdplay->p_vtsi->vts_vobu_admap->vobu_start_sectors[i_vobu]
             <= i_block )
    {
        ++i_vobu;
    }

    dvdplay->state.i_cellN  = i_cell;
    dvdplay->state.i_blockN =
        dvdplay->p_vtsi->vts_vobu_admap->vobu_start_sectors[i_vobu-1] -
        p_pgc->cell_playback[i_cell-1].first_sector;

    dvdplay_nav( dvdplay );

    dvdplay->state.i_blockN = i_block -
        p_pgc->cell_playback[i_cell-1].first_sector;

    _UpdatePGN( dvdplay );

    _dvdplay_dbg( dvdplay, "seeking to block %d (cell %d)", i_block, i_cell );

    return 0;
}

/*****************************************************************************
 * dvdplay_resume: return to saved state.
 *****************************************************************************
 * Resume can be called when a menu has been requested to go back the former
 * play
 *****************************************************************************/
int dvdplay_resume( dvdplay_ptr dvdplay )
{
    link_t  link_values;
    int     i;

     /* Check and see if there is any rsm info!! */
    if( dvdplay->resume.i_vtsN == 0)
    {
        return 0;
    }

    _SetDomain( dvdplay, VTS_DOMAIN );
    _OpenVTSI ( dvdplay, dvdplay->resume.i_vtsN );
    _OpenFile ( dvdplay );
    _SetPGC   ( dvdplay, dvdplay->resume.i_pgcN );

    /* These should never be set in SystemSpace and/or MenuSpace */
    // state.TTN_REG = state.rsm_tt;
    // state.TT_PGCN_REG = state.rsm_pgcN;
    // state.HL_BTNN_REG = state.rsm_btnn;
    for( i = 0 ; i < 5 ; i++ )
    {
        dvdplay->registers.SPRM[4 + i] = dvdplay->pi_rsm_regs[i];
    }

    if( dvdplay->resume.i_cellN == 0)
    {
        if( !dvdplay->state.i_cellN )
        {
            _dvdplay_warn( dvdplay, "state cell is 0" );
        }

        dvdplay->state.i_pgN = 1;
        _PlayPG( dvdplay );
        _ProcessLink( dvdplay );

        if( dvdplay->link.command != PlayThis )
        {
            _dvdplay_warn( dvdplay, "link command is not play (%d)",
                                    dvdplay->link.command );
        }

        dvdplay->state.i_blockN = dvdplay->link.data1;
    }
    else
    {
        dvdplay->state.i_cellN = dvdplay->resume.i_cellN;
        dvdplay->state.i_blockN = dvdplay->resume.i_blockN;
        //state.i_pgN = ?? does this gets the right value in play_Cell, no!
        if( _UpdatePGN( dvdplay ) )
        {
            /* Were at or past the end of the PGC,
             * should not happen for a RSM */
            _dvdplay_warn( dvdplay, "end of PGC during resume" );
            _PlayPGCpost( dvdplay );
        }
    }

    /* Jump */
    dvdplay->b_jump = 1;

    _dvdplay_dbg( dvdplay, "manager state resumed" );

    return 0;
}

/*****************************************************************************
 * dvdplay_button:
 *****************************************************************************/
int dvdplay_button( dvdplay_ptr dvdplay, dvdplay_ctrl_t * p_control )
{
    /* Keep the button register value in a local variable. */
    uint16_t i_btn = dvdplay->HL_BTNN_REG >> 10;

    /* MORE CODE HERE :) */

    /* Paranoia.. */
    if( CELL.first_sector > dvdplay->pci.pci_gi.nv_pck_lbn
         || CELL.last_vobu_start_sector < dvdplay->pci.pci_gi.nv_pck_lbn )
    {
        _dvdplay_err( dvdplay, "cell playback information does not match pci" );

        return -1;
    }

    /* No highlight/button pci info to use */
    if( ( dvdplay->pci.hli.hl_gi.hli_ss & 0x03 ) == 0 )
    {
        _dvdplay_dbg( dvdplay, "no highlight/button pci to use" );

        return 0;
    }

    /* Selected button > than max button ? */
    // FIXME $$$ check how btn_ofn affects things like this
    if( i_btn > dvdplay->pci.hli.hl_gi.btn_ns )
    {
        _dvdplay_dbg( dvdplay, "selected button > max button" );

        i_btn = 1;
    }

    switch( p_control->type )
    {
    case DVDCtrlUpperButtonSelect:
        i_btn = dvdplay->pci.hli.btnit[i_btn - 1].up;
        break;
    case DVDCtrlLowerButtonSelect:
        i_btn = dvdplay->pci.hli.btnit[i_btn - 1].down;
        break;
    case DVDCtrlLeftButtonSelect:
        i_btn = dvdplay->pci.hli.btnit[i_btn - 1].left;
        break;
    case DVDCtrlRightButtonSelect:
        i_btn = dvdplay->pci.hli.btnit[i_btn - 1].right;
        break;
    case DVDCtrlButtonActivate:
        dvdplay->b_action_highlight = 1;
        break;
    case DVDCtrlButtonSelectAndActivate:
        dvdplay->b_action_highlight = 1;
        /* Fall through */
    case DVDCtrlButtonSelect:
        i_btn = p_control->button.i_nr;
        break;
    case DVDCtrlMouseActivate:
        dvdplay->b_action_highlight = 1;
        /* Fall through */
    case DVDCtrlMouseSelect:
    {
        int i_selected_button;

        i_selected_button = MouseButton( &dvdplay->pci, &p_control->mouse );

        if( i_selected_button )
        {
            i_btn = i_selected_button;
        }
        else
        {
            dvdplay->b_action_highlight = 0;
        }
    }
        break;
    default:
        _dvdplay_warn( dvdplay, "ignoring dvdctrl event (%d)",
                                p_control->type );
        break;
    }

    /* Must check if the current selected button has auto_action_mode !!! */
    /* Don't do auto action if it's been selected with the mouse... ?? */
    switch( dvdplay->pci.hli.btnit[i_btn - 1].auto_action_mode )
    {
    case 0:
        break;
    case 1:
        if( p_control->type == DVDCtrlMouseSelect )
        {
            /* auto_action buttons can't be select if they are not activated
             * keep the previous selected button */
            i_btn = dvdplay->HL_BTNN_REG >> 10;
        }
        else
        {
            _dvdplay_warn( dvdplay, "auto_action_mode set!" );

            dvdplay->b_action_highlight = 1;
        }

        break;
    case 2:
    case 3:
    default:
        _dvdplay_err( dvdplay, "unknown auto_action_mode for btn %d", i_btn );
        /* FIXME: should we do something here? */
    }

    /* If a new button has been selected or if one has been activated.
     * Determine the correct area and send the information to the spu
     * decoder. Don't send if its the same as last time. */
    if( dvdplay->b_action_highlight ||
        i_btn != ( dvdplay->HL_BTNN_REG >> 10 ) )
    {
        /* Write the (updated) value back to the button register. */
        dvdplay->HL_BTNN_REG = i_btn << 10;

        dvdplay->pf_callback( dvdplay->p_args, NEW_HIGHLIGHT );
    }
    else
    {
        /* Write the (updated) value back to the button register. */
        dvdplay->HL_BTNN_REG = i_btn << 10;
    }

    if( dvdplay->b_action_highlight )
    {
        _dvdplay_dbg( dvdplay, "executing button command for button %d",
                               i_btn );

        dvdplay_cmd( dvdplay, &dvdplay->pci.hli.btnit[i_btn - 1].cmd );
        dvdplay->b_action_highlight = 0;
        dvdplay->b_jump = 1;
        return 1;
    }

    return 0;
}

/*****************************************************************************
 * dvdplay_highlight
 *****************************************************************************/
int dvdplay_highlight( dvdplay_ptr dvdplay, dvdplay_highlight_t * p_hl )
{
    btni_t *    p_button;
    uint32_t    i_button_coli;
    int         i_button;
    int         i;

    i_button        = dvdplay->HL_BTNN_REG >> 10;
    p_button        = &dvdplay->pci.hli.btnit[i_button - 1];
    i_button_coli   = dvdplay->pci.hli.btn_colit.btn_coli[p_button->btn_coln-1]
            [dvdplay->b_action_highlight];

    p_hl->i_x_start = p_button->x_start;
    p_hl->i_y_start = p_button->y_start;
    p_hl->i_x_end   = p_button->x_end;
    p_hl->i_y_end   = p_button->y_end;

    for( i = 0 ; i < 4 ; i ++ )
    {
        p_hl->pi_color[i]    = 0xf & ( i_button_coli >> ( 16 + 4*i ) );
        p_hl->pi_contrast[i] = 0xf & ( i_button_coli >> ( 4*i ) );
    }

    return 0;
}

/*****************************************************************************
 * dvdplay_cmd: evaluate given command
 *****************************************************************************/
int dvdplay_cmd( dvdplay_ptr dvdplay, vm_cmd_t *cmd )
{
    if( _dvdplay_CommandTable( dvdplay, cmd, 1 ) )
    {
        _ProcessLink( dvdplay );

        if( dvdplay->link.command != PlayThis )
        {
            _dvdplay_warn( dvdplay, "link command is not play (%d)",
                                    dvdplay->link.command );
        }

        dvdplay->state.i_blockN = dvdplay->link.data1;

        /* Something changed, Jump */
        dvdplay->b_jump = 1;
    }

    return 0;
}

/*****************************************************************************
 * dvdplay_menu: try to go to requested menu
 *****************************************************************************/
int dvdplay_menu( dvdplay_ptr     dvdplay,
                         dvdplay_menu_t  menuid,
                         int             i_block )
{
    domain_t    old_domain;
    link_t      link_values;

    /* Should check if we are allowed/can acces this menu */
    _dvdplay_dbg( dvdplay, "jumping to menu %d", menuid );

    /* FIXME XXX $$$ How much state needs to be restored
     * when we fail to find a menu? */
    old_domain = dvdplay->state.domain;

    switch( dvdplay->state.domain )
    {
    case VTS_DOMAIN:
        _SaveRSMinfo( dvdplay, 0, i_block);
        /* FALL THROUGH */
    case VTSM_DOMAIN:
    case VMGM_DOMAIN:
        _SetDomain( dvdplay, _MenuId2Domain( menuid ) );
        if( _GetPGCIT( dvdplay ) != NULL &&
            _SetMenu( dvdplay, menuid ) != -1 )
        {
            _PlayPGC( dvdplay );
            _ProcessLink( dvdplay );

            if( dvdplay->link.command != PlayThis )
            {
                _dvdplay_warn( dvdplay, "link command is not play (%d)",
                                        dvdplay->link.command );
            }

            dvdplay->state.i_blockN = dvdplay->link.data1;

            /* Jump */
            dvdplay->b_jump = 1;
        }
        else
        {
            _SetDomain( dvdplay, old_domain );
        }
        _OpenFile ( dvdplay );

        break;
    case FP_DOMAIN:
        /* FIXME XXX $$$ What should we do here? */
        break;
    }

    return 0;
}

/*****************************************************************************
 * dvdplay_next_cell: jump to next cell ; to be called after each cell end.
 *****************************************************************************/
int dvdplay_next_cell( dvdplay_ptr dvdplay )
{
    _PlayCellPost( dvdplay );
    _ProcessLink( dvdplay );

    if( dvdplay->link.command != PlayThis )
    {
        _dvdplay_warn( dvdplay, "link command is not play (%d)",
                                dvdplay->link.command );
    }

    dvdplay->state.i_blockN = dvdplay->link.data1;

     _dvdplay_dbg( dvdplay, "next cell at block %d", dvdplay->state.i_blockN );

    /* Jump */
    dvdplay->b_jump = 1;

    return 0;
}

/*****************************************************************************
 * dvdplay_pg: go to specified program (often known as chapter).
 *****************************************************************************
 * If i_pg is 0, go back to beginning of current program.
 *****************************************************************************/
int dvdplay_pg( dvdplay_ptr dvdplay, int i_pg )
{
    _dvdplay_dbg( dvdplay, "jumping to program %d", i_pg );

    if( i_pg )
    {
        /* FIXME: check i_pg */
        dvdplay->state.i_pgN = i_pg;
    }

    _PlayPG( dvdplay );
    _ProcessLink( dvdplay );

    if( dvdplay->link.command != PlayThis )
    {
        _dvdplay_warn( dvdplay, "link command is not play (%d)",
                                dvdplay->link.command );
    }

    dvdplay->state.i_blockN = dvdplay->link.data1;

    /* Jump */
    dvdplay->b_jump = 1;

    return 0;
}

/*****************************************************************************
 * dvdplay_angle: select the specified angle and update state for seamless
 * playback.
 *****************************************************************************/
int dvdplay_angle( dvdplay_ptr dvdplay, int i_angle )
{
    _dvdplay_dbg( dvdplay, "selecting angle %d", i_angle );

    if( dvdplay->state.domain == VTS_DOMAIN )
    {
        int i_delta;
        int i_cell;

        if( i_angle == dvdplay->AGL_REG )
        {
            _dvdplay_warn( dvdplay, "angle %d already selected", i_angle );
            return 1;
        }

        i_delta = i_angle - dvdplay->AGL_REG;
        dvdplay->AGL_REG = i_angle;
        if( dvdplay->state.p_pgc->cell_playback[
                dvdplay->state.i_cellN - 1 ].block_mode != 0 )
        {
            if( dvdplay->dsi.sml_agli.data[i_angle - 1].address )
            {
                dvdplay->dsi.vobu_sri.next_vobu =
                    dvdplay->dsi.sml_agli.data[i_angle - 1].address;
            }
            if( dvdplay->dsi.sml_pbi.ilvu_ea )
            {
                dvdplay->dsi.dsi_gi.vobu_ea =
                    dvdplay->dsi.sml_pbi.ilvu_ea;
            }

            i_cell = dvdplay->state.i_cellN + i_delta;

            /* Update state */
            dvdplay->state.i_blockN -=
                dvdplay->state.p_pgc->cell_playback[i_cell-1].first_sector -
                dvdplay->state.p_pgc->cell_playback[
                    dvdplay->state.i_cellN - 1 ].first_sector;
            dvdplay->state.i_cellN = i_cell;
        }

        return 0;

    }

    return 1;
}

/*****************************************************************************
 * dvdplay_audio: select specified audio stream.
 *****************************************************************************/
int dvdplay_audio( dvdplay_ptr dvdplay, int i_audio )
{
    _dvdplay_dbg( dvdplay, "selecting audio %d", i_audio );

    if( i_audio == dvdplay->AST_REG )
    {
        _dvdplay_warn( dvdplay, "audio %d, already selected", i_audio );
        return -1;
    }

    dvdplay->AST_REG = i_audio;

    return 0;
}

/*****************************************************************************
 * dvdplay_subp: select specified subpicture stream.
 *****************************************************************************/
int dvdplay_subp( dvdplay_ptr dvdplay, int i_subp )
{
    _dvdplay_dbg( dvdplay, "selecting subp %d", i_subp );

    if( i_subp == dvdplay->SPST_REG )
    {
        _dvdplay_warn( dvdplay, "subo %d, already selected", i_subp );
        return -1;
    }

    dvdplay->SPST_REG = i_subp & 0x40;

    return 0;
}


/*
 * local functions
 */

/*****************************************************************************
 * ProcessPCI: update hilighted button and send info to spu decoder.
 *****************************************************************************/
static int ProcessPCI( dvdplay_ptr dvdplay )
{
    hli_t *     p_hli;
    btni_t *    p_btn;
    uint16_t    i_btn_nr;

    p_hli    = &dvdplay->pci.hli;
    i_btn_nr = dvdplay->HL_BTNN_REG >> 10;

    /* Check if this is alright, i.e. pci->hli.hl_gi.hli_ss == 1 only
     * for the first menu pic packet? Should be.
     * What about looping menus? Will reset it every loop.. */
    if( p_hli->hl_gi.hli_ss == 1 )
    {
        if( p_hli->hl_gi.fosl_btnn != 0)
        {
            i_btn_nr = p_hli->hl_gi.fosl_btnn;
            _dvdplay_warn( dvdplay, "forced select button %d",
                                    p_hli->hl_gi.fosl_btnn );
        }
    }

    /* Paranoia.. */
    if( ( p_hli->hl_gi.hli_ss & 0x03 ) != 0
     && i_btn_nr > p_hli->hl_gi.btn_ns )
    {
        i_btn_nr = 1;
    }

    /* FIXME TODO XXX $$$ */

    /* Determine the correct area and send the information
     * to the spu decoder. */
    /* Possible optimization: don't send if its the same as last time.
     * same, as in same hli info, same button number and same
     * select/action state
     * Note this send a highlight even if hli_ss == 0, it then turns the
     * highlight off. */

    p_btn = &p_hli->btnit[i_btn_nr - 1];

    /* TODO XXX */
#if 0
    send_highlight(button->x_start, button->y_start,
                   button->x_end, button->y_end,
                   pci->hli.btn_colit.btn_coli[button->btn_coln-1][0]);
#endif

    /* Write the (updated) value back to the button register. */
    dvdplay->HL_BTNN_REG = i_btn_nr << 10;
    dvdplay->pf_callback( dvdplay->p_args, NEW_HIGHLIGHT );

#if 0
    {
        int i;
        for( i=0;i<p_hli->hl_gi.btn_ns;++i)
        {
            fprintf(stderr, "button %d: (%d,%d)-(%d,%d)\n", i+1,
                    p_hli->btnit[i].x_start,
                    p_hli->btnit[i].y_start,
                    p_hli->btnit[i].x_end,
                    p_hli->btnit[i].y_end );
        }
    }
#endif

    return 0;
}

/*****************************************************************************
 * MouseButton
 *****************************************************************************/
static int MouseButton( pci_t * p_pci, dvdplay_ctrl_mouse_t * p_mouse )
{
    int     i_button;
    int     i_x = p_mouse->i_x;
    int     i_y = p_mouse->i_y;

    for( i_button = 1 ; i_button <= p_pci->hli.hl_gi.btn_ns ; ++i_button )
    {
        if( ( i_x >= p_pci->hli.btnit[i_button-1].x_start )
             && ( i_x <= p_pci->hli.btnit[i_button-1].x_end )
         && ( i_y >= p_pci->hli.btnit[i_button-1].y_start )
         && ( i_y <= p_pci->hli.btnit[i_button-1].y_end ) )
        {
            return i_button;
        }
    }

    return 0;
}

/*****************************************************************************
 * NextDataPacket: parse block until the next start code and return next
 * packet length
 *****************************************************************************/
static int NextDataPacket( const dvdplay_ptr dvdplay, byte_t** pp_buffer )
{
    byte_t* p_ptr;
    int     i_packet_size;

    p_ptr = *pp_buffer;

    while( p_ptr[0] || p_ptr[1] || p_ptr[2] != 1 || p_ptr[3] < 0xB9 )
    {
        if( p_ptr >= *pp_buffer + DVD_VIDEO_LB_LEN )
        {
            _dvdplay_err( dvdplay, "cannot find start code in nav packet" );
            return -1;
        }

        p_ptr++;
    }

    /* Taken from vlc */
    /* 0x1B9 == SYSTEM_END_CODE, it is only 4 bytes long. */
    if( p_ptr[3] != 0xB9 )
    {
        if( p_ptr[3] != 0xBA )
        {
            /* That's the case for all packets, except pack header. */
            i_packet_size = (p_ptr[4] << 8) | p_ptr[5];
        }
        else
        {
            /* Pack header. */
            if( (p_ptr[4] & 0xC0) == 0x40 )
            {
                /* MPEG-2 */
                i_packet_size = 8;
            }
            else if( (p_ptr[4] & 0xF0) == 0x20 )
            {
                /* MPEG-1 */
                i_packet_size = 6;
            }
            else
            {
                fprintf( stderr, "***** Unable to determine stream type\n" );
                return( -1 );
            }
        }
    }
    else
    {
        /* System End Code */
        i_packet_size = -2;
    }

    *pp_buffer = p_ptr + 6;

    return i_packet_size;
}


/*****************************************************************************
 * ReadNav: extract PCI and DSI packets from navigation block.
 *****************************************************************************/
static int ReadNav( dvdplay_ptr dvdplay, byte_t* p_buffer )
{
    byte_t*     p_ptr;
    int         i_packet_size;

    dvdplay->pci.pci_gi.nv_pck_lbn = -1;
    dvdplay->dsi.dsi_gi.nv_pck_lbn = -1;

    p_ptr = p_buffer;

    while( p_ptr < p_buffer + DVD_VIDEO_LB_LEN )
    {
        if( ( i_packet_size = NextDataPacket( dvdplay, &p_ptr ) ) < 0 )
        {
            fprintf( stderr, "***** cannot find data packet\n" );
            return -1;
        }

        if( *p_ptr == PS2_PCI_SUBSTREAM_ID && i_packet_size == PCI_BYTES )
        {
            navRead_PCI( &dvdplay->pci, p_ptr + 1);
            ProcessPCI ( dvdplay );
        }
        else if( *p_ptr == PS2_DSI_SUBSTREAM_ID && i_packet_size == DSI_BYTES )
        {
            navRead_DSI( &dvdplay->dsi, p_ptr + 1 );
        }

        p_ptr += i_packet_size;
    }

    return 0;
}

#undef CELL
